"""Builds an adversarial training network."""

# pylint: disable=too-many-arguments
# pylint: disable=too-many-locals
# pylint: disable=too-many-statements
# pylint: disable=no-self-use
# pylint: disable=too-few-public-methods

from caffe.proto import caffe_pb2
from caffe import Net

from google.protobuf import text_format

# pylint: disable=invalid-name
# pylint: disable=no-member
LayerType = caffe_pb2.LayerParameter.LayerType
EltwiseOp = caffe_pb2.EltwiseParameter.EltwiseOp
PoolMethod = caffe_pb2.PoolingParameter.PoolMethod
DBType = caffe_pb2.DataParameter.DB
# pylint: enable=invalid-name
# pylint: enable=no-member

batchsize = 1;

class NetworkBuilder(object):

  def __init__(self, training_batch_size, testing_batch_size, **kwargs):
    self.training_batch_size = training_batch_size
    self.testing_batch_size = testing_batch_size
    self.other_args = kwargs


  def _make_inception(self, network, x1x1, x3x3r, x3x3, x5x5r, x5x5, proj,
                      name_generator):
    """Make Inception submodule."""

    layers = []

    split = self._make_split_layer(network)
    layers.append(split)

    context1 = self._make_conv_layer(network, kernel_size=1, num_output=x1x1,
                                     bias_value=0)
    layers.append(context1)

    relu1 = self._make_relu_layer(network)
    layers.append(relu1)

    context2a = self._make_conv_layer(network, kernel_size=1, num_output=x3x3r,
                                      bias_value=0)
    layers.append(context2a)

    relu2a = self._make_relu_layer(network)
    layers.append(relu2a)

    context2b = self._make_conv_layer(network, kernel_size=3, num_output=x3x3,
                                      pad=1)
    layers.append(context2b)

    relu2b = self._make_relu_layer(network)
    layers.append(relu2b)

    context3a = self._make_conv_layer(network, kernel_size=1, num_output=x5x5r,
                                      bias_value=0)
    layers.append(context3a)

    relu3a = self._make_relu_layer(network)
    layers.append(relu3a)

    context3b = self._make_conv_layer(network, kernel_size=5, num_output=x5x5,
                                      pad=2)
    layers.append(context3b)

    relu3b = self._make_relu_layer(network)
    layers.append(relu3b)

    context4a = self._make_maxpool_layer(network, kernel_size=3)
    layers.append(context4a)

    relu4a = self._make_relu_layer(network)
    layers.append(relu4a)

    context4b = self._make_conv_layer(network, kernel_size=1, num_output=proj,
                                      pad=1, bias_value=0)
    layers.append(context4b)

    relu4b = self._make_relu_layer(network)
    layers.append(relu4b)

    concat = self._make_concat_layer(network)
    layers.append(concat)

    connections = [
      (split.name, (split.top, context1.bottom)),
      (split.name, (split.top, context2a.bottom)),
      (split.name, (split.top, context3a.bottom)),
      (split.name, (split.top, context4a.bottom)),
      (context2a.name,
          (context2a.top, relu2a.bottom, relu2a.top, context2b.bottom)),
      (context3a.name,
          (context3a.top, relu3a.bottom, relu3a.top, context3b.bottom)),
      (context4a.name,
          (context4a.top, relu4a.bottom, relu4a.top, context4b.bottom)),
      (context1.name, (context1.top, relu1.bottom, relu1.top, concat.bottom)),
      (context2b.name,
          (context2b.top, relu2b.bottom, relu2b.top, concat.bottom)),
      (context3b.name,
          (context3b.top, relu3b.bottom, relu3b.top, concat.bottom)),
      (context4b.name,
          (context4b.top, relu4b.bottom, relu4b.top, concat.bottom)),
    ]

    for connection in connections:
      self._tie(connection, name_generator)

    return layers

  def _make_sum_layer(self, network):
    layer = network.layers.add()
    layer.name = 'sum'
    layer.type = LayerType.Value('ELTWISE')
    params = layer.eltwise_param
    params.operation = EltwiseOp.Value('SUM')
    return layer

  def _make_upsampling_layer(self, network, stride):
    layer = network.layers.add()
    layer.name = 'upsample'
    layer.type = LayerType.Value('UPSAMPLING')
    params = layer.upsampling_param
    params.kernel_size = stride
    return layer

  def _make_folding_layer(self, network, channels, height, width, prefix=''):
    layer = network.layers.add()
    layer.name = '%sfolding' % (prefix)
    layer.type = LayerType.Value('FOLDING')
    params = layer.folding_param
    params.channels_folded = channels
    params.height_folded = height
    params.width_folded = width
    return layer

  def _make_conv_layer(self, network, kernel_size, num_output, stride=1, pad=0,
                       bias_value=0.1, shared_name=None, wtype='xavier', std=0.01):
    """Make convolution layer."""

    layer = network.layers.add()
    layer.name = 'conv_%dx%d_%d' % (kernel_size, kernel_size, stride)

    layer.type = LayerType.Value('CONVOLUTION')
    params = layer.convolution_param
    params.num_output = num_output
    params.kernel_size = kernel_size
    params.stride = stride
    params.pad = pad
    weight_filler = params.weight_filler
    weight_filler.type = wtype
    if weight_filler.type == 'gaussian':
      weight_filler.mean = 0
      weight_filler.std = std
    bias_filler = params.bias_filler
    bias_filler.type = 'constant'
    bias_filler.value = bias_value

    layer.blobs_lr.append(1)
    layer.blobs_lr.append(2)

    layer.weight_decay.append(1)
    layer.weight_decay.append(0)

    if shared_name:
      layer.param.append('%s_w' % shared_name)
      layer.param.append('%s_b' % shared_name)

    return layer

  def _make_maxpool_layer(self, network, kernel_size, stride=1):
    """Make max pooling layer."""

    layer = network.layers.add()
    layer.name = 'maxpool_%dx%d_%d' % (kernel_size, kernel_size, stride)

    layer.type = LayerType.Value('POOLING')
    params = layer.pooling_param
    params.pool = PoolMethod.Value('MAX')
    params.kernel_size = kernel_size
    params.stride = stride

    return layer

  def _make_avgpool_layer(self, network, kernel_size, stride=1):
    """Make average pooling layer."""

    layer = network.layers.add()
    layer.name = 'avgpool_%dx%d_%d' % (kernel_size, kernel_size, stride)

    layer.type = LayerType.Value('POOLING')
    params = layer.pooling_param
    params.pool = PoolMethod.Value('AVE')
    params.kernel_size = kernel_size
    params.stride = stride

    return layer

  def _make_lrn_layer(self, network, name='lrn'):
    """Make local response normalization layer."""

    layer = network.layers.add()
    layer.name = name

    layer.type = LayerType.Value('LRN')
    params = layer.lrn_param
    params.local_size = 5
    params.alpha = 0.0001
    params.beta = 0.75

    return layer

  def _make_concat_layer(self, network):
    """Make depth concatenation layer."""

    layer = network.layers.add()
    layer.name = 'concat'

    layer.type = LayerType.Value('CONCAT')

    return layer

  def _make_dropout_layer(self, network, dropout_ratio=0.5):
    """Make dropout layer."""

    layer = network.layers.add()
    layer.name = 'dropout'

    layer.type = LayerType.Value('DROPOUT')
    params = layer.dropout_param
    params.dropout_ratio = dropout_ratio

    return layer

  def _make_inner_product_layer(self, network, num_output, weight_lr=1,
                                bias_lr=2, bias_value=0.1, prefix='',
                                wtype='xavier', std=0.01):
    """Make inner product layer."""

    layer = network.layers.add()
    layer.name = '%sinner_product' % prefix

    layer.type = LayerType.Value('INNER_PRODUCT')
    params = layer.inner_product_param
    params.num_output = num_output
    weight_filler = params.weight_filler
    weight_filler.type = wtype
    if wtype == 'gaussian':
      weight_filler.mean = 0
      weight_filler.std = std
    bias_filler = params.bias_filler
    bias_filler.type = 'constant'
    bias_filler.value = bias_value

    layer.blobs_lr.append(weight_lr)
    layer.blobs_lr.append(bias_lr)

    layer.weight_decay.append(1)
    layer.weight_decay.append(0)

    return layer

  def _make_split_layer(self, network):
    """Make split layer."""

    layer = network.layers.add()
    layer.name = 'split'

    layer.type = LayerType.Value('SPLIT')

    return layer

  def _make_relu_layer(self, network):
    """Make ReLU layer."""

    layer = network.layers.add()
    layer.name = 'relu'

    layer.type = LayerType.Value('RELU')

    return layer

  def _tie(self, layers, name_generator):
    """Generate a named connection between layer endpoints."""

    name = 'ep_%s_%d' % (layers[0], name_generator.next())
    for layer in layers[1]:
      layer.append(name)

  def _connection_name_generator(self):
    """Generate a unique id."""

    index = 0
    while True:
      yield index
      index += 1

  def _make_generator32_v1(self, network, name_generator, numstep=1, wtype='xavier', wf_std=0.001):
    layers = []

    # fc layers.
    fc1 = self._make_inner_product_layer(network, 1024, wtype=wtype, std=wf_std) 
    fc1.bottom.append('sample')
    layers.append(fc1)

    relu1 = self._make_relu_layer(network)
    layers.append(relu1)

    fc2 = self._make_inner_product_layer(network, 4096, wtype=wtype, std=wf_std)
    layers.append(fc2)

    # folding layer -> 8x8x64.
    folding = self._make_folding_layer(network, channels=64, height=8, width=8)
    layers.append(folding)

    conv1 = []
    relu1 = []
    deconv1 = []
    upsample1 = []
    unpool1 = []
    conv2 = []
    relu2 = []
    deconv2 = []
    upsample2 = []
    unpool2 = []

    # unpooling layer -> 16x16x64.
    upsample1.append(self._make_upsampling_layer(network, stride=2))

    upsample1_split = self._make_split_layer(network)
    layers.append(upsample1_split)

    # First step has no bottom-up.
    unpool1 = [ upsample1[0] ]

    # Relu
    unpool1_relu.append(self._make_relu_layer(network))

    # top-down conv -> 16x16x64
    conv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=64, pad=2, bias_value=0.1,
                                       shared_name='conv1', wtype=wtype,
                                       std=wf_std))

    # conv1[-1] is now 16x16x64.
    # unpooling layer -> 32x32x64.
    upsample2.append(self._make_upsampling_layer(network, stride=2))

    # Add bottom-up and top-down.
    unpool2 = [ upsample2[0] ]

    # Relu
    unpool2_relu.append(self._make_relu_layer(network))

    # top-down conv -> 32x32x3
    conv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=3, pad=2, bias_value=0.1,
                                       shared_name='conv2', wtype=wtype,
                                       std=wf_std))

    for s in range(numstep):
      # deconv1 -> 16x16x64 from previous time step.
      deconv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv1', wtype=wtype,
                                           std=wf_std))

      # Updated unpooling units for time s+1.
      unpool1.append(self._make_sum_layer(network))

      # Apply relu.
      unpool1_relu.append(self._make_relu_layer(network))

      # conv1.
      conv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=64, pad=2, bias_value=0.1,
                                         shared_name='conv1', wtype=wtype,
                                         std=wf_std))

      # deconv2 -> 32x32x64 from previous time step.
      deconv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv2', wtype=wtype,
                                           std=wf_std))

      # Add bottom-up and top-down.
      unpool2.append(self._make_sum_layer(network))

      # Relu
      unpool2_relu.append(self._make_relu_layer(network))

      # top-down conv -> 32x32x64
      conv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=3, pad=2, bias_value=0.1,
                                         shared_name='conv2', wtype=wtype,
                                         std=wf_std))

    # 32x32 side head.
    conv2[-1].top.append('conv32')

    layers += unpool1
    layers += upsample1
    layers += deconv1
    layers += unpool1_relu
    layers += conv1
    layers += unpool2
    layers += upsample2
    layers += deconv2
    layers += unpool2_relu
    layers += conv2

    # Upsample 1.
    connections = [
      (fc1.name, (fc1.top, relu1.bottom, relu1.top, fc2.bottom)),
      (fc2.name, (fc2.top, folding.bottom)),
      (folding.name, (folding.top, upsample1[0].bottom)),
      (upsample1[0].name, (upsample1[0].top, unpool1[0].bottom)),
      (unpool1[0].name, (unpool1[0].top, upsample1_split.bottom))
    ]
    for s in range(numstep):
      # upsample1 -> unpool1.
      connections.append((upsample1_split.name, (upsample1_split.top, unpool1[s+1].bottom)))
      # deconv1 -> unpool1.
      connections.append((deconv1[s].name, (deconv1[s].top, unpool1[s+1].bottom)))
      # unpool1 -> unpool1_relu.
      connections.append((unpool1[s+1].name, (unpool1[s+1].top, unpool1_relu[s+1].bottom)))
      # unpool1_relu -> conv1.
      connections.append((unpool1_relu[s+1].name, (unpool1_relu[s+1].top, conv1[s+1].bottom)))

      # conv1 -> upsample2.
      connections.append((conv1[s+1].name, (conv1[s+1].top, upsample2[s+1].bottom)))
      # upsample2 -> unpool2.
      connections.append((upsample2[s+1].name, (upsample2[s+1].top, unpool2[s+1].bottom)))
      # deconv2 -> unpool2.
      connections.append((deconv2[s].name, (deconv2[s].top, unpool2[s+1].bottom)))
      # unpool2 -> unpool2_relu.
      connections.append((unpool2[s+1].name, (unpool2[s+1].top, unpool2_relu[s+1].bottom)))
      # unpool2_relu -> conv2.
      connections.append((unpool2_relu[s+1].name, (unpool2_relu[s+1].top, conv2[s+1].bottom)))

    for connection in connections:
      self._tie(connection, name_generator)
    return layers

  def _make_generator32(self, network, name_generator, numstep=0, wtype='xavier', wf_std=0.001):
    layers = []

    # fc layers.
    fc1 = self._make_inner_product_layer(network, 1024, wtype=wtype, std=wf_std) 
    fc1.bottom.append('sample')
    layers.append(fc1)

    relu1 = self._make_relu_layer(network)
    layers.append(relu1)

    fc2 = self._make_inner_product_layer(network, 4096, wtype=wtype, std=wf_std)
    layers.append(fc2)

    # folding layer -> 8x8x64.
    folding = self._make_folding_layer(network, channels=64, height=8, width=8)
    layers.append(folding)

    # unpooling layer -> 16x16x64.
    upsample1 = self._make_upsampling_layer(network, stride=2)
    layers.append(upsample1)

    unpool1 = [] 
    unpool1.append(self._make_split_layer(network))

    unpool1_relu = []
    unpool1_relu.append(self._make_relu_layer(network))

    conv1 = []
    conv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=64, pad=2, bias_value=0.1,
                                       shared_name='conv1', wtype=wtype,
                                       std=wf_std))
    deconv1 = []

    for s in range(numstep):
      # bottom-up conv -> 16x16x64
      deconv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv1', wtype=wtype,
                                           std=wf_std))

      # Add bottom-up and top-down.
      unpool1.append(self._make_sum_layer(network))

      # Relu
      unpool1_relu.append(self._make_relu_layer(network))

      # top-down conv -> 16x16x64
      conv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=64, pad=2, bias_value=0.1,
                                         shared_name='conv1', wtype=wtype,
                                         std=wf_std))

    layers += unpool1
    layers += deconv1
    layers += unpool1_relu
    layers += conv1

    # conv1[-1] is now 16x16x64.
    # unpooling layer -> 32x32x64.
    upsample2 = self._make_upsampling_layer(network, stride=2)
    layers.append(upsample2)

    unpool2 = [] 
    unpool2.append(self._make_split_layer(network))

    unpool2_relu = []
    unpool2_relu.append(self._make_relu_layer(network))

    conv2 = []
    conv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=3, pad=2, bias_value=0.1,
                                       shared_name='conv2', wtype=wtype,
                                       std=wf_std))

    deconv2 = []

    for s in range(numstep):
      # bottom-up conv -> 32x32x64
      deconv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv2', wtype=wtype,
                                           std=wf_std))

      # Add bottom-up and top-down.
      unpool2.append(self._make_sum_layer(network))

      # Relu
      unpool2_relu.append(self._make_relu_layer(network))

      # top-down conv -> 32x32x64
      conv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=3, pad=2, bias_value=0.1,
                                         shared_name='conv2', wtype=wtype,
                                         std=wf_std))

    # 32x32 side head.
    conv2[-1].top.append('conv32')

    layers += unpool2
    layers += conv2
    layers += unpool2_relu
    layers += deconv2

    # Upsample 1.
    connections = [
      (fc1.name, (fc1.top, relu1.bottom, relu1.top, fc2.bottom)),
      (fc2.name, (fc2.top, folding.bottom)),
      (folding.name, (folding.top, upsample1.bottom)),
      (upsample1.name, (upsample1.top, unpool1[0].bottom)),
      (unpool1[0].name, (unpool1[0].top, unpool1_relu[0].bottom)),
      (unpool1_relu[0].name, (unpool1_relu[0].top, conv1[0].bottom))
    ]
    for s in range(numstep):
      connections.append((conv1[s].name, (conv1[s].top, deconv1[s].bottom)))
      # bottom-up connection from time-s convolution.
      connections.append((deconv1[s].name, (deconv1[s].top, unpool1[s+1].bottom)))
      # top-down connection from time-0 unpooling.
      connections.append((unpool1[0].name, (unpool1[0].top, unpool1[s+1].bottom)))
      # relu -> conv1.
      connections.append((unpool1[s+1].name, (unpool1[s+1].top, unpool1_relu[s+1].bottom, unpool1_relu[s+1].top, conv1[s+1].bottom)))
    # Upsample 2.
    connections += [
      (conv1[-1].name, (conv1[-1].top, upsample2.bottom)),
      (upsample2.name, (upsample2.top, unpool2[0].bottom)),
      (unpool2[0].name, (unpool2[0].top, unpool2_relu[0].bottom)),
      (unpool2_relu[0].name, (unpool2_relu[0].top, conv2[0].bottom))
    ]
    for s in range(numstep):
      connections.append((conv2[s].name, (conv2[s].top, deconv2[s].bottom)))
      # bottom-up connection from time-s convolution.
      connections.append((deconv2[s].name, (deconv2[s].top, unpool2[s+1].bottom)))
      # top-down connection from time-0 unpooling.
      connections.append((unpool2[0].name, (unpool2[0].top, unpool2[s+1].bottom)))
      # relu -> conv2.
      connections.append((unpool2[s+1].name, (unpool2[s+1].top, unpool2_relu[s+1].bottom, unpool2_relu[s+1].top, conv2[s+1].bottom)))

    for connection in connections:
      self._tie(connection, name_generator)
    return layers

  def _make_generator64(self, network, name_generator, numstep=1, wtype='xavier', wf_std=0.001):
    layers = []

    # fc layers.
    fc1 = self._make_inner_product_layer(network, 1024, wtype=wtype, std=wf_std) 
    fc1.bottom.append('sample')
    layers.append(fc1)

    relu1 = self._make_relu_layer(network)
    layers.append(relu1)

    fc2 = self._make_inner_product_layer(network, 4096, wtype=wtype, std=wf_std)
    layers.append(fc2)

    # folding layer -> 8x8x64.
    folding = self._make_folding_layer(network, channels=64, height=8, width=8)
    layers.append(folding)

    # unpooling layer -> 16x16x64.
    upsample1 = self._make_upsampling_layer(network, stride=2)
    layers.append(upsample1)

    unpool1 = [] 
    unpool1.append(self._make_split_layer(network))

    unpool1_relu = []
    unpool1_relu.append(self._make_relu_layer(network))

    conv1 = []
    conv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=64, pad=2, bias_value=0.1,
                                       shared_name='conv1', wtype=wtype,
                                       std=wf_std))
    deconv1 = []

    for s in range(numstep):
      # bottom-up conv -> 16x16x64
      deconv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv1', wtype=wtype,
                                           std=wf_std))

      # Add bottom-up and top-down.
      unpool1.append(self._make_sum_layer(network))

      # Relu
      unpool1_relu.append(self._make_relu_layer(network))

      # top-down conv -> 16x16x64
      conv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=64, pad=2, bias_value=0.1,
                                         shared_name='conv1', wtype=wtype,
                                         std=wf_std))

    layers += unpool1
    layers += deconv1
    layers += unpool1_relu
    layers += conv1

    # conv1[-1] is now 16x16x64.
    # unpooling layer -> 32x32x64.
    upsample2 = self._make_upsampling_layer(network, stride=2)
    layers.append(upsample2)

    unpool2 = [] 
    unpool2.append(self._make_split_layer(network))

    unpool2_relu = []
    unpool2_relu.append(self._make_relu_layer(network))

    conv2 = []
    conv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=64, pad=2, bias_value=0.1,
                                       shared_name='conv2', wtype=wtype,
                                       std=wf_std))

    deconv2 = []

    for s in range(numstep):
      # bottom-up conv -> 32x32x64
      deconv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv2', wtype=wtype,
                                           std=wf_std))

      # Add bottom-up and top-down.
      unpool2.append(self._make_sum_layer(network))

      # Relu
      unpool2_relu.append(self._make_relu_layer(network))

      # top-down conv -> 32x32x64
      conv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=64, pad=2, bias_value=0.1,
                                         shared_name='conv2', wtype=wtype,
                                         std=wf_std))

    layers += unpool2
    layers += conv2
    layers += unpool2_relu
    layers += deconv2

    # conv2[-1] is now 32x32x64.
    # unpooling layer -> 64x64x64.
    upsample3 = self._make_upsampling_layer(network, stride=2)
    layers.append(upsample3)

    unpool3 = [] 
    unpool3.append(self._make_split_layer(network))

    unpool3_relu = []
    unpool3_relu.append(self._make_relu_layer(network))

    conv3 = []
    conv3.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=64, pad=2, bias_value=0.1,
                                       shared_name='conv3', wtype=wtype,
                                       std=wf_std))

    deconv3 = []

    for s in range(numstep):
      # bottom-up conv -> 64x64x64
      deconv3.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv3', wtype=wtype,
                                           std=wf_std))

      # Add bottom-up and top-down.
      unpool3.append(self._make_sum_layer(network))

      # Relu
      unpool3_relu.append(self._make_relu_layer(network))

      # top-down conv -> 64x64x64
      conv3.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=64, pad=2, bias_value=0.1,
                                         shared_name='conv3', wtype=wtype,
                                         std=wf_std))

    # 64x64 side head.
    conv64_head = self._make_conv_layer(network, kernel_size=5, stride = 1,
                                        num_output=3, pad=2, bias_value=0.0,
                                        wtype='xavier', std=0.01)
    conv64_head.top.append('conv64')
    layers.append(conv64_head)

    layers += unpool3
    layers += conv3
    layers += unpool3_relu
    layers += deconv3

    # Upsample 1.
    connections = [
      (fc1.name, (fc1.top, relu1.bottom, relu1.top, fc2.bottom)),
      (fc2.name, (fc2.top, folding.bottom)),
      (folding.name, (folding.top, upsample1.bottom)),
      (upsample1.name, (upsample1.top, unpool1[0].bottom)),
      (unpool1[0].name, (unpool1[0].top, unpool1_relu[0].bottom)),
      (unpool1_relu[0].name, (unpool1_relu[0].top, conv1[0].bottom))
    ]
    for s in range(numstep):
      connections.append((conv1[s].name, (conv1[s].top, deconv1[s].bottom)))
      # bottom-up connection from time-s convolution.
      connections.append((deconv1[s].name, (deconv1[s].top, unpool1[s+1].bottom)))
      # top-down connection from time-0 unpooling.
      connections.append((unpool1[0].name, (unpool1[0].top, unpool1[s+1].bottom)))
      # relu -> conv1.
      connections.append((unpool1[s+1].name, (unpool1[s+1].top, unpool1_relu[s+1].bottom, unpool1_relu[s+1].top, conv1[s+1].bottom)))
    # Upsample 2.
    connections += [
      (conv1[-1].name, (conv1[-1].top, upsample2.bottom)),
      (upsample2.name, (upsample2.top, unpool2[0].bottom)),
      (unpool2[0].name, (unpool2[0].top, unpool2_relu[0].bottom)),
      (unpool2_relu[0].name, (unpool2_relu[0].top, conv2[0].bottom))
    ]
    for s in range(numstep):
      connections.append((conv2[s].name, (conv2[s].top, deconv2[s].bottom)))
      # bottom-up connection from time-s convolution.
      connections.append((deconv2[s].name, (deconv2[s].top, unpool2[s+1].bottom)))
      # top-down connection from time-0 unpooling.
      connections.append((unpool2[0].name, (unpool2[0].top, unpool2[s+1].bottom)))
      # relu -> conv2.
      connections.append((unpool2[s+1].name, (unpool2[s+1].top, unpool2_relu[s+1].bottom, unpool2_relu[s+1].top, conv2[s+1].bottom)))
    # Upsample 3.
    connections += [
      (conv2[-1].name, (conv2[-1].top, upsample3.bottom)),
      (upsample3.name, (upsample3.top, unpool3[0].bottom)),
      (unpool3[0].name, (unpool3[0].top, unpool3_relu[0].bottom)),
      (unpool3_relu[0].name, (unpool3_relu[0].top, conv3[0].bottom))
    ]
    for s in range(numstep):
      connections.append((conv3[s].name, (conv3[s].top, deconv3[s].bottom)))
      # bottom-up connection from time-s convolution.
      connections.append((deconv3[s].name, (deconv3[s].top, unpool3[s+1].bottom)))
      # top-down connection from time-0 unpooling.
      connections.append((unpool3[0].name, (unpool3[0].top, unpool3[s+1].bottom)))
      # relu -> conv3.
      connections.append((unpool3[s+1].name, (unpool3[s+1].top, unpool3_relu[s+1].bottom, unpool3_relu[s+1].top, conv3[s+1].bottom)))

    connections.append((conv3[-1].name, (conv3[-1].top, conv64_head.bottom)))

    for connection in connections:
      self._tie(connection, name_generator)
    return layers


  def _make_generator(self, network, name_generator, numstep=1, wtype='xavier', wf_std=0.001):
    layers = []

    # fc layers.
    fc1 = self._make_inner_product_layer(network, 1024, wtype=wtype, std=wf_std) 
    fc1.bottom.append('sample')
    layers.append(fc1)

    relu1 = self._make_relu_layer(network)
    layers.append(relu1)

    fc2 = self._make_inner_product_layer(network, 4096, wtype=wtype, std=wf_std)
    layers.append(fc2)

    # folding layer -> 8x8x64.
    folding = self._make_folding_layer(network, channels=64, height=8, width=8)
    layers.append(folding)

    # unpooling layer -> 16x16x64.
    upsample1 = self._make_upsampling_layer(network, stride=2)
    layers.append(upsample1)

    unpool1 = [] 
    unpool1.append(self._make_split_layer(network))

    unpool1_relu = []
    unpool1_relu.append(self._make_relu_layer(network))

    conv1 = []
    conv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=64, pad=2, bias_value=0.1,
                                       shared_name='conv1', wtype=wtype,
                                       std=wf_std))
    deconv1 = []

    for s in range(numstep):
      # bottom-up conv -> 16x16x64
      deconv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv1', wtype=wtype,
                                           std=wf_std))

      # Add bottom-up and top-down.
      unpool1.append(self._make_sum_layer(network))

      # Relu
      unpool1_relu.append(self._make_relu_layer(network))

      # top-down conv -> 16x16x64
      conv1.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=64, pad=2, bias_value=0.1,
                                         shared_name='conv1', wtype=wtype,
                                         std=wf_std))

    layers += unpool1
    layers += deconv1
    layers += unpool1_relu
    layers += conv1

    # conv1[-1] is now 16x16x64.
    # unpooling layer -> 32x32x64.
    upsample2 = self._make_upsampling_layer(network, stride=2)
    layers.append(upsample2)

    unpool2 = [] 
    unpool2.append(self._make_split_layer(network))

    unpool2_relu = []
    unpool2_relu.append(self._make_relu_layer(network))

    conv2 = []
    conv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=64, pad=2, bias_value=0.1,
                                       shared_name='conv2', wtype=wtype,
                                       std=wf_std))

    deconv2 = []

    for s in range(numstep):
      # bottom-up conv -> 32x32x64
      deconv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv2', wtype=wtype,
                                           std=wf_std))

      # Add bottom-up and top-down.
      unpool2.append(self._make_sum_layer(network))

      # Relu
      unpool2_relu.append(self._make_relu_layer(network))

      # top-down conv -> 32x32x64
      conv2.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=64, pad=2, bias_value=0.1,
                                         shared_name='conv2', wtype=wtype,
                                         std=wf_std))

    # 32x32 side head.
    conv2_out = self._make_split_layer(network)
    layers.append(conv2_out)

    conv32_head = self._make_conv_layer(network, kernel_size=5, stride = 1,
                                        num_output=3, pad=2, bias_value=0.0,
                                        wtype='xavier', std=0.1)
    conv32_head.top.append('conv32')
    layers.append(conv32_head)

    layers += unpool2
    layers += conv2
    layers += unpool2_relu
    layers += deconv2

    # conv2[-1] is now 32x32x64.
    # unpooling layer -> 64x64x64.
    upsample3 = self._make_upsampling_layer(network, stride=2)
    layers.append(upsample3)

    unpool3 = [] 
    unpool3.append(self._make_split_layer(network))

    unpool3_relu = []
    unpool3_relu.append(self._make_relu_layer(network))

    conv3 = []
    conv3.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=64, pad=2, bias_value=0.1,
                                       shared_name='conv3', wtype=wtype,
                                       std=wf_std))

    deconv3 = []

    for s in range(numstep):
      # bottom-up conv -> 64x64x64
      deconv3.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv3', wtype=wtype,
                                           std=wf_std))

      # Add bottom-up and top-down.
      unpool3.append(self._make_sum_layer(network))

      # Relu
      unpool3_relu.append(self._make_relu_layer(network))

      # top-down conv -> 64x64x64
      conv3.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=64, pad=2, bias_value=0.1,
                                         shared_name='conv3', wtype=wtype,
                                         std=wf_std))

    # 64x64 side head.
    conv3_out = self._make_split_layer(network)
    layers.append(conv3_out)

    conv64_head = self._make_conv_layer(network, kernel_size=5, stride = 1,
                                        num_output=3, pad=2, bias_value=0.0,
                                        wtype='xavier', std=0.01)
    conv64_head.top.append('conv64')
    layers.append(conv64_head)

    layers += unpool3
    layers += conv3
    layers += unpool3_relu
    layers += deconv3

    # conv3[-1] is now 64x64x64.
    # unpooling layer -> 128x128x64.
    upsample4 = self._make_upsampling_layer(network, stride=2)
    layers.append(upsample4)

    unpool4 = [] 
    unpool4.append(self._make_split_layer(network))

    unpool4_relu = []
    unpool4_relu.append(self._make_relu_layer(network))

    conv4 = []
    conv4.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                       num_output=3, pad=2, bias_value=0.1,
                                       shared_name='conv4', wtype=wtype,
                                       std=wf_std))

    deconv4 = []

    for s in range(numstep):
      # bottom-up conv -> 128x128x64
      deconv4.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                           num_output=64, pad=2, bias_value=0.1,
                                           shared_name='deconv4', wtype=wtype,
                                           std=wf_std))

      # Add bottom-up and top-down.
      unpool4.append(self._make_sum_layer(network))

      # Relu
      unpool4_relu.append(self._make_relu_layer(network))

      # top-down conv -> 128x128x64
      conv4.append(self._make_conv_layer(network, kernel_size=5, stride=1,
                                         num_output=3, pad=2, bias_value=0.1,
                                         shared_name='conv4', wtype=wtype,
                                         std=wf_std))

    #conv128_head = self._make_conv_layer(network, kernel_size=5, stride = 1,
    #                                     num_output=3, pad=2, bias_value=0.0,
    #                                     wtype='gaussian', std=0.1)
    #conv128_head.top.append('conv128')
    #layers.append(conv128_head)
    conv4[-1].top.append('conv128')
  
    layers += unpool4
    layers += conv4
    layers += unpool4_relu
    layers += deconv4

    # Upsample 1.
    connections = [
      (fc1.name, (fc1.top, relu1.bottom, relu1.top, fc2.bottom)),
      (fc2.name, (fc2.top, folding.bottom)),
      (folding.name, (folding.top, upsample1.bottom)),
      (upsample1.name, (upsample1.top, unpool1[0].bottom)),
      (unpool1[0].name, (unpool1[0].top, unpool1_relu[0].bottom)),
      (unpool1_relu[0].name, (unpool1_relu[0].top, conv1[0].bottom))
    ]
    for s in range(numstep):
      connections.append((conv1[s].name, (conv1[s].top, deconv1[s].bottom)))
      # bottom-up connection from time-s convolution.
      connections.append((deconv1[s].name, (deconv1[s].top, unpool1[s+1].bottom)))
      # top-down connection from time-0 unpooling.
      connections.append((unpool1[0].name, (unpool1[0].top, unpool1[s+1].bottom)))
      # relu -> conv1.
      connections.append((unpool1[s+1].name, (unpool1[s+1].top, unpool1_relu[s+1].bottom, unpool1_relu[s+1].top, conv1[s+1].bottom)))
    # Upsample 2.
    connections += [
      (conv1[-1].name, (conv1[-1].top, upsample2.bottom)),
      (upsample2.name, (upsample2.top, unpool2[0].bottom)),
      (unpool2[0].name, (unpool2[0].top, unpool2_relu[0].bottom)),
      (unpool2_relu[0].name, (unpool2_relu[0].top, conv2[0].bottom))
    ]
    for s in range(numstep):
      connections.append((conv2[s].name, (conv2[s].top, deconv2[s].bottom)))
      # bottom-up connection from time-s convolution.
      connections.append((deconv2[s].name, (deconv2[s].top, unpool2[s+1].bottom)))
      # top-down connection from time-0 unpooling.
      connections.append((unpool2[0].name, (unpool2[0].top, unpool2[s+1].bottom)))
      # relu -> conv2.
      connections.append((unpool2[s+1].name, (unpool2[s+1].top, unpool2_relu[s+1].bottom, unpool2_relu[s+1].top, conv2[s+1].bottom)))
    # Upsample 3.
    connections += [
      (conv2[-1].name, (conv2[-1].top, conv2_out.bottom)),
      (conv2_out.name, (conv2_out.top, conv32_head.bottom)), # attach side head.
      (conv2_out.name, (conv2_out.top, upsample3.bottom)),
      (upsample3.name, (upsample3.top, unpool3[0].bottom)),
      (unpool3[0].name, (unpool3[0].top, unpool3_relu[0].bottom)),
      (unpool3_relu[0].name, (unpool3_relu[0].top, conv3[0].bottom))
    ]
    for s in range(numstep):
      connections.append((conv3[s].name, (conv3[s].top, deconv3[s].bottom)))
      # bottom-up connection from time-s convolution.
      connections.append((deconv3[s].name, (deconv3[s].top, unpool3[s+1].bottom)))
      # top-down connection from time-0 unpooling.
      connections.append((unpool3[0].name, (unpool3[0].top, unpool3[s+1].bottom)))
      # relu -> conv3.
      connections.append((unpool3[s+1].name, (unpool3[s+1].top, unpool3_relu[s+1].bottom, unpool3_relu[s+1].top, conv3[s+1].bottom)))
    # Upsample 4.
    connections += [
      (conv3[-1].name, (conv3[-1].top, conv3_out.bottom)),
      (conv3_out.name, (conv3_out.top, conv64_head.bottom)), # attach side head.
      (conv3_out.name, (conv3_out.top, upsample4.bottom)),
      (upsample4.name, (upsample4.top, unpool4[0].bottom)),
      (unpool4[0].name, (unpool4[0].top, unpool4_relu[0].bottom)),
      (unpool4_relu[0].name, (unpool4_relu[0].top, conv4[0].bottom))
    ]
    for s in range(numstep):
      connections.append((conv4[s].name, (conv4[s].top, deconv4[s].bottom)))
      # bottom-up connection from time-s convolution.
      connections.append((deconv4[s].name, (deconv4[s].top, unpool4[s+1].bottom)))
      # top-down connection from time-0 unpooling.
      connections.append((unpool4[0].name, (unpool4[0].top, unpool4[s+1].bottom)))
      # relu -> conv4.
      connections.append((unpool4[s+1].name, (unpool4[s+1].top, unpool4_relu[s+1].bottom, unpool4_relu[s+1].top, conv4[s+1].bottom)))
    #connections.append((conv4[-1].name, (conv4[-1].top, conv128_head.bottom)))

    for connection in connections:
      self._tie(connection, name_generator)
    return layers

  def _build_generator64_network(self):
    """Build the Generator network."""

    network = caffe_pb2.NetParameter()
    network.force_backward = True
    network.name = 'generator'
    network.input.append('sample')
    #network.input_dim.append(20)
    #network.input_dim.append(10)
    network.input_dim.append(1)
    network.input_dim.append(100)
    #network.input_dim.append(512)
    network.input_dim.append(1)
    network.input_dim.append(1)

    layers = []

    name_generator = self._connection_name_generator()

    # Conv and FC layers will share params among columns.
    param_types = [ LayerType.Value('INNER_PRODUCT'),
                    LayerType.Value('CONVOLUTION') ]

    # fully-connected layers.
    gen_layers = self._make_generator64(network, name_generator, numstep=2)
    layers += gen_layers

    # Fix up the names based on the connections that were generated.
    for pos, layer in enumerate(layers):
      layer.name += '_%d' % pos

    return network

  def _build_generator32_network(self):
    """Build the Generator network."""

    network = caffe_pb2.NetParameter()
    network.force_backward = True
    network.name = 'generator'
    network.input.append('sample')
    #network.input_dim.append(20)
    network.input_dim.append(10)
    network.input_dim.append(100)
    #network.input_dim.append(512)
    network.input_dim.append(1)
    network.input_dim.append(1)

    layers = []

    name_generator = self._connection_name_generator()

    # Conv and FC layers will share params among columns.
    param_types = [ LayerType.Value('INNER_PRODUCT'),
                    LayerType.Value('CONVOLUTION') ]

    # fully-connected layers.
    gen_layers = self._make_generator32(network, name_generator, numstep=2)
    layers += gen_layers

    # Fix up the names based on the connections that were generated.
    for pos, layer in enumerate(layers):
      layer.name += '_%d' % pos

    return network

  def _build_generator_network(self):
    """Build the Generator network."""

    network = caffe_pb2.NetParameter()
    network.force_backward = True
    network.name = 'generator'
    network.input.append('sample')
    network.input_dim.append(10)
    #network.input_dim.append(100)
    network.input_dim.append(100)
    network.input_dim.append(1)
    network.input_dim.append(1)

    layers = []

    name_generator = self._connection_name_generator()

    # Conv and FC layers will share params among columns.
    param_types = [ LayerType.Value('INNER_PRODUCT'),
                    LayerType.Value('CONVOLUTION') ]

    # fully-connected layers.
    gen_layers = self._make_generator(network, name_generator, numstep=2)
    layers += gen_layers

    # Fix up the names based on the connections that were generated.
    for pos, layer in enumerate(layers):
      layer.name += '_%d' % pos

    return network

  def _build_discriminator32_network(self, wtype='xavier', std=0.01):
    """Build the 32x32x3 discriminator network."""

    network = caffe_pb2.NetParameter()
    network.force_backward = True
    network.name = 'discriminator32'
    network.input.append('data')
    #network.input_dim.append(40)
    #network.input_dim.append(120)
    network.input_dim.append(20)
    network.input_dim.append(3)
    network.input_dim.append(32)
    network.input_dim.append(32)

    layers = []
    name_generator = self._connection_name_generator()

    # 32x32x3.
    conv1 = self._make_conv_layer(network, kernel_size=5, stride=1,
                                  num_output=64, pad=2, bias_value=0.1,
                                  wtype=wtype, std=std)
    conv1.bottom.append('data')
    layers.append(conv1)

    relu1 = self._make_relu_layer(network)
    layers.append(relu1)

    pool1 = self._make_maxpool_layer(network, kernel_size=2, stride=2)
    layers.append(pool1)

    # 16x16x64
    conv2 = self._make_conv_layer(network, kernel_size=3, stride=1,
                                  num_output=72, pad=1, bias_value=0.1,
                                  wtype=wtype, std=std)
    layers.append(conv2)

    relu2 = self._make_relu_layer(network)
    layers.append(relu2)

    pool2 = self._make_maxpool_layer(network, kernel_size=2, stride=2)
    layers.append(pool2)

    # 8x8x72
    conv3 = self._make_conv_layer(network, kernel_size=3, stride=1,
                                  num_output=32, pad=0, bias_value=0.1,
                                  wtype=wtype, std=std)
    layers.append(conv3)

    relu3 = self._make_relu_layer(network)
    layers.append(relu3)

    # 6x6x32 -> 1
    fc1 = self._make_inner_product_layer(network, num_output=1)
    fc1.top.append('prediction')
    layers.append(fc1)

    connections = [
      (conv1.name, (conv1.top, relu1.bottom, relu1.top, pool1.bottom)),
      (pool1.name, (pool1.top, conv2.bottom)),
      (conv2.name, (conv2.top, relu2.bottom, relu2.top, pool2.bottom)),
      (pool2.name, (pool2.top, conv3.bottom)),
      (conv3.name, (conv3.top, relu3.bottom, relu3.top, fc1.bottom))
    ]

    # make connections.
    for connection in connections:
      self._tie(connection, name_generator)

    # Fix up the names based on the connections that were generated.
    for pos, layer in enumerate(layers):
      layer.name += '_%d' % pos

    return network

  def _build_discriminator64_network(self, wtype='xavier', std=0.01):
    """Build the 64x64x3 discriminator network."""

    network = caffe_pb2.NetParameter()
    network.force_backward = True
    network.name = 'discriminator64'
    network.input.append('data')
    #network.input_dim.append(20)
    network.input_dim.append(2)
    network.input_dim.append(3)
    network.input_dim.append(64)
    network.input_dim.append(64)

    layers = []
    name_generator = self._connection_name_generator()

    # 64x64x3.
    conv1 = self._make_conv_layer(network, kernel_size=5, stride=1,
                                  num_output=64, pad=2, bias_value=0.1,
                                  wtype=wtype, std=std)
    conv1.bottom.append('data')
    layers.append(conv1)

    relu1 = self._make_relu_layer(network)
    layers.append(relu1)

    pool1 = self._make_maxpool_layer(network, kernel_size=2, stride=2)
    layers.append(pool1)

    # 32x32x64
    conv2 = self._make_conv_layer(network, kernel_size=5, stride=1,
                                  num_output=72, pad=2, bias_value=0.1,
                                  wtype=wtype, std=std)
    layers.append(conv2)

    relu2 = self._make_relu_layer(network)
    layers.append(relu2)

    pool2 = self._make_maxpool_layer(network, kernel_size=2, stride=2)
    layers.append(pool2)

    # 16x16x72
    conv3 = self._make_conv_layer(network, kernel_size=5, stride=1,
                                  num_output=72, pad=2, bias_value=0.1,
                                  wtype=wtype, std=std)
    layers.append(conv3)

    relu3 = self._make_relu_layer(network)
    layers.append(relu3)

    pool3 = self._make_maxpool_layer(network, kernel_size=2, stride=2)
    layers.append(pool3)

    # 8x8x72
    conv4 = self._make_conv_layer(network, kernel_size=3, stride=1,
                                  num_output=32, pad=0, bias_value=0.1,
                                  wtype=wtype, std=std)
    layers.append(conv4)

    relu4 = self._make_relu_layer(network)
    layers.append(relu4)

    # 6x6x32 -> 1
    fc1 = self._make_inner_product_layer(network, num_output=1)
    fc1.top.append('prediction')
    layers.append(fc1)

    connections = [
      (conv1.name, (conv1.top, relu1.bottom, relu1.top, pool1.bottom)),
      (pool1.name, (pool1.top, conv2.bottom)),
      (conv2.name, (conv2.top, relu2.bottom, relu2.top, pool2.bottom)),
      (pool2.name, (pool2.top, conv3.bottom)),
      (conv3.name, (conv3.top, relu3.bottom, relu3.top, pool3.bottom)),
      (pool3.name, (pool3.top, conv4.bottom)),
      (conv4.name, (conv4.top, relu4.bottom, relu4.top, fc1.bottom))
    ]

    # make connections.
    for connection in connections:
      self._tie(connection, name_generator)

    # Fix up the names based on the connections that were generated.
    for pos, layer in enumerate(layers):
      layer.name += '_%d' % pos

    return network

  def _build_discriminator128_network(self, wtype='xavier', std=0.01):
    """Build the 128x128x3 discriminator network."""

    network = caffe_pb2.NetParameter()
    network.force_backward = True
    network.name = 'discriminator128'
    network.input.append('data')
    network.input_dim.append(20)
    network.input_dim.append(3)
    network.input_dim.append(128)
    network.input_dim.append(128)

    layers = []
    name_generator = self._connection_name_generator()

    # 128x128x3.
    conv1 = self._make_conv_layer(network, kernel_size=7, stride=1,
                                  num_output=64, pad=3, bias_value=0.1,
                                  wtype=wtype, std=std)
    conv1.bottom.append('data')
    layers.append(conv1)

    relu1 = self._make_relu_layer(network)
    layers.append(relu1)

    pool1 = self._make_maxpool_layer(network, kernel_size=2, stride=2)
    layers.append(pool1)

    # 64x64x64.
    conv2 = self._make_conv_layer(network, kernel_size=5, stride=1,
                                  num_output=64, pad=2, bias_value=0.1,
                                  wtype=wtype, std=std)
    layers.append(conv2)

    relu2 = self._make_relu_layer(network)
    layers.append(relu2)

    pool2 = self._make_maxpool_layer(network, kernel_size=2, stride=2)
    layers.append(pool2)

    # 32x32x64
    conv3 = self._make_conv_layer(network, kernel_size=5, stride=1,
                                  num_output=72, pad=2, bias_value=0.1,
                                  wtype=wtype, std=std)
    layers.append(conv3)

    relu3 = self._make_relu_layer(network)
    layers.append(relu3)

    pool3 = self._make_maxpool_layer(network, kernel_size=2, stride=2)
    layers.append(pool3)

    # 16x16x72
    conv4 = self._make_conv_layer(network, kernel_size=5, stride=1,
                                  num_output=72, pad=2, bias_value=0.1,
                                  wtype=wtype, std=std)
    layers.append(conv4)

    relu4 = self._make_relu_layer(network)
    layers.append(relu4)

    pool4 = self._make_maxpool_layer(network, kernel_size=2, stride=2)
    layers.append(pool4)

    # 8x8x72
    conv5 = self._make_conv_layer(network, kernel_size=3, stride=1,
                                  num_output=32, pad=0, bias_value=0.1,
                                  wtype=wtype, std=std)
    layers.append(conv5)

    relu5 = self._make_relu_layer(network)
    layers.append(relu5)

    # 6x6x32 -> 1
    fc1 = self._make_inner_product_layer(network, num_output=1,
                                  wtype=wtype, std=std)
    fc1.top.append('prediction')
    layers.append(fc1)

    connections = [
      (conv1.name, (conv1.top, relu1.bottom, relu1.top, pool1.bottom)),
      (pool1.name, (pool1.top, conv2.bottom)),
      (conv2.name, (conv2.top, relu2.bottom, relu2.top, pool2.bottom)),
      (pool2.name, (pool2.top, conv3.bottom)),
      (conv3.name, (conv3.top, relu3.bottom, relu3.top, pool3.bottom)),
      (pool3.name, (pool3.top, conv4.bottom)),
      (conv4.name, (conv4.top, relu4.bottom, relu4.top, pool4.bottom)),
      (pool4.name, (pool4.top, conv5.bottom)),
      (conv5.name, (conv5.top, relu5.bottom, relu5.top, fc1.bottom))
    ]

    # make connections.
    for connection in connections:
      self._tie(connection, name_generator)

    # Fix up the names based on the connections that were generated.
    for pos, layer in enumerate(layers):
      layer.name += '_%d' % pos

    return network


  def build_network(self, netname):
    """main method."""

    if netname == 'generator':
      network = self._build_generator_network()
    elif netname == 'generator32':
      network = self._build_generator32_network()
    elif netname == 'generator64':
      network = self._build_generator64_network()
    elif netname == 'discriminator32':
      network = self._build_discriminator32_network()
    elif netname == 'discriminator64':
      network = self._build_discriminator64_network()
    elif netname == 'discriminator128':
      network = self._build_discriminator128_network()
    else:
      print('unknown netname: %s' % netname)
      return

    network_filename = '%s.prototxt' % netname
    print network
    with open(network_filename, 'w') as network_file:
      network_file.write(text_format.MessageToString(network))
    return Net(network_filename)


if __name__ == '__main__':
  __Network_builder__ = NetworkBuilder(
    training_batch_size=20,
    testing_batch_size=20,
  )

  #__Network_builder__.build_network('generator')
  __Network_builder__.build_network('generator32')
  #__Network_builder__.build_network('generator64')
  #__Network_builder__.build_network('discriminator32')
  #__Network_builder__.build_network('discriminator64')
  #__Network_builder__.build_network('discriminator128')

