function demo_shapes_figures_videos(gpu_id, expname)

% init script
if ~exist('gpu_id','var') || isempty(gpu_id)
    gpu_id = 1;
end

script_dir = fileparts(mfilename('fullpath'));

addpath(fullfile(script_dir ,'../../matlab/caffe'));
addpath(fullfile(script_dir ,'../util'));

export_fig_dir = fullfile(script_dir,'../export_fig-master');
if ~exist(export_fig_dir,'dir')
  fprintf( 'Fetching export_fig : ' ); tic
  urlwrite('https://github.com/altmany/export_fig/archive/master.zip',[export_fig_dir '.zip']);
  unzip( [export_fig_dir '.zip'], fullfile(script_dir,'..') );
  delete([export_fig_dir '.zip']);
  toc
end
addpath(export_fig_dir);

if ~exist('expname','var') || isempty(expname)
    expname='deep';
end

expnames={'add','mul','deep'};

assert(ismember(expname,expnames),'Invalid expname');

fprintf( '============ DEMO: %s\n', expname );
demo_shapes_internal(gpu_id, expname);

function demo_shapes_internal (gpu_id, expname)

imgsize = [ 48 48 ];
numhid = 512;

% search for trained model
BASE_DIR = '.';
switch expname
    case 'add'
        init_from = 'analogy_shapes_add.mat';
    case 'deep'
        init_from = 'analogy_shapes_deep.mat';
    case 'mul'
        init_from = 'analogy_shapes_mul.mat';
end

% init cache dir
subdir = sprintf('shapes_%s', expname);
solver_enc = 'analogy_enc_solver.prototxt';
solver_enc = fullfile(BASE_DIR,sprintf('results/%s/%s', subdir, solver_enc));
solver_dec = 'analogy_dec_solver.prototxt';
solver_dec = fullfile(BASE_DIR,sprintf('results/%s/%s', subdir, solver_dec));
init_from  = fullfile(BASE_DIR,sprintf('results/%s', subdir),init_from);
fprintf('* Enc:   %s\n', solver_enc);
fprintf('* Dec:   %s\n', solver_dec);
fprintf('* Model: %s\n', init_from);

% load data

data_cache_fn = '../data/shapes48.mat';
if ~exist( data_cache_fn, 'file' )
    mkdir_p( fileparts(data_cache_fn) );
    fprintf( 'Generate shape data: ' ); tic
    [M,L] = gen_shapes_data;
    save(data_cache_fn,'M','L');
    toc
end
load(data_cache_fn);
fprintf( 'Load shape data: ' ); tic
data = load(data_cache_fn);
data = single(data.M);
toc

% load training model

fprintf( 'Load model: ' ); tic
model = load(init_from);
pars = model.pars;
matcaffe_init_shapes_train(gpu_id, ... 
        sprintf(solver_enc, subdir), ... 
        sprintf(solver_dec, subdir), ... 
        init_from);
caffe('set_phase_test');
toc

% Make val pairs.
rng('default');
rng(1);
[tmp1,tmp2] = find(~pars.pairs);
valtestpairs = [ tmp1, tmp2 ];
Rval = randsample(size(valtestpairs,1), 30);
Rtest = setdiff(1:size(valtestpairs,1), Rval);
valpairs = valtestpairs(Rval,:);
testpairs = valtestpairs(Rtest,:);

%% Make paper figure rows.

HID={};GT={};

%% Def: Rotation.
fprintf( 'Prep rot : ' ); ct = tic; 
pairs = testpairs(92,:);
id1 = pairs(1); id2 = pairs(2);
sz = [ 8, 4 ];
[ col1, shape1 ] = ind2sub(sz,id1);
[ col2, shape2 ] = ind2sub(sz,id2);
rot1 = 4; scale1 = 3; xpos1 = 3; ypos1 = 2;
rot2 = 6; scale2 = 3; xpos2 = 3; ypos2 = 2;
rot3 = 2; scale3 = 2; xpos3 = 2; ypos3 = 3;

ref = repmat(data(:,:,:,col1,shape1,scale1,rot1,xpos1,ypos1),[1 1 1 pars.batchsize]);
out = repmat(data(:,:,:,col1,shape1,scale2,rot2,xpos2,ypos2),[1 1 1 pars.batchsize]);
query = repmat(data(:,:,:,col2,shape2,scale3,rot3,xpos3,ypos3),[1 1 1 pars.batchsize]);
res = caffe('forward', { ref; out; query; 0*ref }, 0);
hid_out = res{1};
hid_query = res{2};
hid_ref = res{3};
GT = [GT;{ref,out,query}];
HID = [HID;{hid_ref,hid_out,hid_query}];
toc(ct);

% Def: Scaling.
fprintf( 'Prep scal : ' ); ct = tic; 
pairs = testpairs(121,:);
id1 = pairs(1); id2 = pairs(2);
[ col1, shape1 ] = ind2sub(sz,id1);
[ col2, shape2 ] = ind2sub(sz,id2);
rot1 = 5; scale1 = 3; xpos1 = 2; ypos1 = 4;
rot2 = 5; scale2 = 2; xpos2 = 2; ypos2 = 4;
rot3 = 2; scale3 = 4; xpos3 = 3; ypos3 = 2;
ref = repmat(data(:,:,:,col1,shape1,scale1,rot1,xpos1,ypos1),[1 1 1 pars.batchsize]);
out = repmat(data(:,:,:,col1,shape1,scale2,rot2,xpos2,ypos2),[1 1 1 pars.batchsize]);
query = repmat(data(:,:,:,col2,shape2,scale3,rot3,xpos3,ypos3),[1 1 1 pars.batchsize]);
res = caffe('forward', { ref; out; query; 0*ref }, 0);
hid_out = res{1};
hid_query = res{2};
hid_ref = res{3};
GT = [GT;{ref,out,query}];
HID = [HID;{hid_ref,hid_out,hid_query}];
toc(ct);

% Def: Translation.
fprintf( 'Prep trans : ' ); ct = tic; 
pairs = testpairs(171,:);
id1 = pairs(1); id2 = pairs(2);
[ col1, shape1 ] = ind2sub(sz,id1);
[ col2, shape2 ] = ind2sub(sz,id2);
rot1 = 5; scale1 = 2; xpos1 = 3; ypos1 = 4;
rot2 = 5; scale2 = 2; xpos2 = 3; ypos2 = 2;
rot3 = 2; scale3 = 2; xpos3 = 1; ypos3 = 5;
ref = repmat(data(:,:,:,col1,shape1,scale1,rot1,xpos1,ypos1),[1 1 1 pars.batchsize]);
out = repmat(data(:,:,:,col1,shape1,scale2,rot2,xpos2,ypos2),[1 1 1 pars.batchsize]);
query = repmat(data(:,:,:,col2,shape2,scale3,rot3,xpos3,ypos3),[1 1 1 pars.batchsize]);
res = caffe('forward', { ref; out; query; 0*ref }, 0);
hid_out = res{1};
hid_query = res{2};
hid_ref = res{3};
GT = [GT;{ref,out,query}];
HID = [HID;{hid_ref,hid_out,hid_query}];
toc(ct);

% Generate figures for different interleaving style
% 1 - rotation, 2 - scaling, 3 - translation
TR_NAME = {'rot','scl', 'trans'};
GIVEN_ID = {1,2,3,[2,3],[1,3],[2,1],[1,2,3]};
QUERY_ID = [1,2,3,2,1,3,1];
BACKWARD_ID = [1,1,1,0,0,0,0];
vis_interleaved = cell(length(QUERY_ID),1);
for t = 1:length(QUERY_ID)
    ename=cellfun( @(s) ['-' s], TR_NAME(GIVEN_ID{t}), 'UniformOutput', 0 );
    ename=[ename{:}]; ename = ename(2:end);
    fprintf( 'Result %d (%s) : ', t, ename ); ct = tic; 
    hid_ref   = HID(GIVEN_ID{t},1); ref   = GT(GIVEN_ID{t},1);
    hid_out   = HID(GIVEN_ID{t},2); out   = GT(GIVEN_ID{t},2); 
    hid_query = HID{QUERY_ID(t),3}; query = GT{QUERY_ID(t),3};
    nstep = numel(hid_ref)*4*(BACKWARD_ID(t)+1)+1;
    vis = analogy_repeated(expname,pars,hid_ref,hid_out,hid_query,nstep, BACKWARD_ID(t));

    vis = cat(4,query(:,:,:,1),vis(:,:,:,2:nstep));
    for j = numel(hid_ref):-1:1
        vis = cat(4,ref{j}(:,:,:,1),out{j}(:,:,:,1),vis);
    end
    vis_interleaved{t} = vis;
    
    minQ = min(query(:)); maxQ = max(query(:));
    vis1 = max(0,vis - minQ);
    vis1 = vis1./(maxQ-minQ);
    vis5 = vis1;
    
    vis1 = permute(vis1,[1 2 4 3]);
    vis1 = reshape(vis1,[pars.vdim,pars.vdim*size(vis1,3),3]);
    imwrite( vis1, sprintf('results/%s/png-%d-%s.png', subdir, t, ename) );
    disp_scale = 2;
    vis2 = imresize(vis1,disp_scale,'nearest'); 
    vis2(end+16,:,:)=1;
    vis2(end-15:end,:,:)=1;
    figure(1); clf; imshow(vis2);
    
    set(gcf, 'Color', 'white');
    
    hold on
    labels = {};
    for j = 1:numel(hid_ref)
        labels = [labels;{sprintf( 'ref%d', j )}; 
            {['+' TR_NAME{GIVEN_ID{t}(j)} ' (gt)']}];
    end
    labels = [labels; {'query'}];
    pr_labels = rrepmat( TR_NAME(GIVEN_ID{t}), [1, nstep-1] );
    for i=1:length(pr_labels)
        if BACKWARD_ID(t) && i>length(pr_labels)/2
            pr_labels{i} = ['-',pr_labels{i}];
        else
            pr_labels{i} = ['+',pr_labels{i}];
        end
    end
    labels = [labels; pr_labels.'];
    for i = 1:length(labels);
        line( (i*pars.vdim*disp_scale+0.5)*[1 1],[0 pars.vdim*disp_scale+1], [0 0], 'color', 'white' );
        text( (i-0.5)*pars.vdim*disp_scale, pars.vdim*disp_scale+1, labels{i}, ...
            'HorizontalAlignment', 'center', 'VerticalAlignment', 'top' )
    end
    line( [0 0],[0 pars.vdim*disp_scale+1], [0 0], 'color', 'white' );
    pause(0.1);
    export_fig( sprintf('results/%s/fig-%d-%s.pdf', subdir, t, ename) );
    hold off
    
    video_folder = sprintf('results/%s/vid-%d-%s', subdir, t, ename);
    if ~exist( video_folder, 'dir' )
        mkdir(video_folder);
    end
    
    writerObj = VideoWriter([video_folder '.avi']);
    writerObj.FrameRate = 2;
    open(writerObj);
    for j = 1:size(vis5,4)
        tmp_vis = vis5(:,:,:,j);
        tmp_vis = imresize(tmp_vis,4,'nearest');
        tmp_vis(end+32,:,:)=1;
        tmp_vis(end-32:end,:,:)=1;
        tmp_vis = max(0,tmp_vis);tmp_vis = min(1,tmp_vis);
        tmp_vis = insertText(tmp_vis, [ size(tmp_vis,2)/2, size(tmp_vis,2)+4 ], labels{j}, ...
            'FontSize',16, 'TextColor', 'Black', 'BoxColor', 'white', 'AnchorPoint', 'CenterTop' );
        imwrite( tmp_vis, fullfile(video_folder,sprintf('%03d.png',j)) );
        writeVideo(writerObj,tmp_vis);
    end
    close(writerObj);
    toc(ct);
end


fname_row = fullfile( sprintf('results/%s',subdir), sprintf('shapes_hard_%s_rows.mat', expname) );
save(fname_row, 'vis_interleaved');

