% Load the TFD data and extract OMP features.
%
% xlab          : labeled image features.
% labs_ex       : emotion labels.
% labs_id       : identity labels.
% folds_train   : folds for training disBM. mirrored, removed bad IDs,
%                   keep only when both emotion and identity labels exist.
% folds_ex      : folds for emotion recognition. mirrored for training and
%                   validation, but not for testing. bad IDs included
% folds_id      : folds for identity verification. not mirrored for all.
%                   bad IDs included.
% xunlab: unlabeled image features.

function [ xlab, labs_ex, labs_id, folds_train, folds_ex, folds_id, ...
           xunlab ] = load_tfd_omp(tfd_path)

% If OMP features have not been extracted yet, extract them and
% save them to disk.
if ~exist('TFD_OMP.mat', 'file'),
    disp('OMP Features not found. Generating OMP features...');
    omp_fea = load('omp_tfd48_l1.mat');
    data = load('TFD_48x48.mat');
    
    images = data.images;
    data = rmfield(data, 'images');
    
    % Remove data with all zeros
    remove_idx = sum(sum(images.^2, 2), 3) == 0;
    images(remove_idx, :, :) = [];
    data.labs_ex(remove_idx) = [];
    data.labs_id(remove_idx) = [];
    data.folds(remove_idx, :) = [];
    
    [ ~, dim1, dim2 ] = size(images);
    
    % Mirror the labeled images.
    images_lab = images(data.labs_ex ~= -1, :, :);
    images_flipped = flipdim(images_lab, 3);
    images_unlab = images(data.labs_ex == -1, :, :);
    
    images_lab = double(reshape(permute(images_lab, [2 3 1]), ...
        [dim1*dim2, numel(images_lab)/(dim1*dim2)]));
    images_flipped = double(reshape(permute(images_flipped, [2 3 1]), ...
        [dim1*dim2, numel(images_flipped)/(dim1*dim2)]));
    images_unlab = double(reshape(permute(images_unlab, [2 3 1]), ...
        [dim1*dim2, numel(images_unlab)/(dim1*dim2)]));
    
    % Extract OMP features. This can take an hour or so.
    features_lab = extract_omp(images_lab', omp_fea.dictionary, omp_fea.rfSize, ...
        [48 48], omp_fea.M, omp_fea.P, omp_fea.encoder, omp_fea.encParam);
    features_flipped = extract_omp(images_flipped', omp_fea.dictionary, omp_fea.rfSize, ...
        [48 48], omp_fea.M, omp_fea.P, omp_fea.encoder, omp_fea.encParam);
    features_unlab = extract_omp(images_unlab', omp_fea.dictionary, omp_fea.rfSize, ...
        [48 48], omp_fea.M, omp_fea.P, omp_fea.encoder, omp_fea.encParam);
    
    features = [features_lab' features_flipped' features_unlab'];
    features = single(features);
    clear features_lab features_flipped features_unlab;
    
    % Now that OMP features are loaded, generate fold indices for training and evaluation.
    id_info = load('tfd_bad_ids.mat');
    badids = id_info.badids;
    
    % Replicate labels and indices due to mirroring.
    % (labeled set for emotion includes labeled set for ids)
    labs_ex_lab = data.labs_ex(data.labs_ex ~= -1);
    labs_ex_flipped = data.labs_ex(data.labs_ex ~= -1);
    labs_ex_unlab = data.labs_ex(data.labs_ex == -1);
    labs_id_lab = data.labs_id(data.labs_ex ~= -1);
    labs_id_flipped = data.labs_id(data.labs_ex ~= -1);
    labs_id_unlab = data.labs_id(data.labs_ex == -1);
    folds_lab = data.folds(data.labs_ex ~= -1, :);
    folds_flipped = data.folds(data.labs_ex ~= -1, :);
    folds_unlab = data.folds(data.labs_ex == -1, :);
    folds_flipped(folds_flipped == 3) = 0; % remove mirrored examples from the test example
    
    % Concatenate
    labs_ex = [ labs_ex_lab ; labs_ex_flipped ; labs_ex_unlab ];
    labs_id = [ labs_id_lab ; labs_id_flipped ; labs_id_unlab ];
    folds_ex = [ folds_lab ; folds_flipped ; folds_unlab ];
    folds_id = [ folds_lab ; 0*folds_flipped ; folds_unlab ];
    
    % Remove suspected bad IDs from training folds.
    for b = 1:length(badids),
        labs_id(labs_id==badids(b)) = -1;
    end

    folds_train = [folds_lab ; folds_flipped ; folds_unlab];
    
    TFD = struct('features', features, 'labs_ex', labs_ex, 'labs_id', labs_id, ...
        'folds_train', folds_train, 'folds_ex', folds_ex, 'folds_id', folds_id);
    save('-v7.3','TFD_OMP.mat', '-struct', 'TFD');
else
    TFD = load('TFD_OMP.mat');
end


labs_ex = TFD.labs_ex;
labs_id = TFD.labs_id;
folds_train = TFD.folds_train;
folds_ex = TFD.folds_ex;
folds_id = TFD.folds_id;
xlab = TFD.features(:, labs_ex ~= -1);
xunlab = TFD.features(:, labs_ex == -1);
clear TFD;

% Rename all IDs so they are contiguous 1,...,n.
unique_ids = unique(labs_id);
unique_ids((unique_ids == -1)) = [];
for i = 1:length(unique_ids),
    labs_id(labs_id==unique_ids(i)) = i;
end

return;
